#include <stdio.h>
#include <string.h>
#include <dos.h>

#define I2CINT     0xAA
#define TCPIPINT   0xAC

#define EEPROM_SLAVE 0xC0

/****************************************************************************/
//Registerstructs
/****************************************************************************/

static union  REGS  inregs;
static union  REGS  outregs;
static struct SREGS sregs;

/****************************************************************************/
//API_funcs
/****************************************************************************/

void api_sleep (unsigned int sleeptime)
{
  //sleep
  inregs.h.ah = 9;
  inregs.x.bx = sleeptime;
  int86(TCPIPINT,&inregs,&outregs);
}

/****************************************************************************/
//I2C_funcs
/****************************************************************************/

void I2C_init (char speed)  //0-3
{
  inregs.h.ah = 0x80;
  if((speed < 0) || (speed >3 ))
  {
	 inregs.h.al = 0;
  }
  else
  {
	 inregs.h.al = speed;
  }
  int86(I2CINT,&inregs,&outregs);
}


void I2C_release (void)
{
	inregs.h.ah = 0x84;
	inregs.h.al = 0;
	int86(I2CINT,&inregs,&outregs);
}


int I2C_scan (unsigned char start_addr, unsigned char end_addr)
{
	inregs.h.ah = 0x81;
	inregs.h.al = start_addr;
	inregs.h.cl = end_addr;
	int86(I2CINT,&inregs,&outregs);
	return outregs.h.al;
}


int I2C_transmit_block (unsigned char slave, char far * buffer, int length)
{
	inregs.h.ah = 0x83;
	inregs.h.al = slave & 0xFE;
	inregs.x.cx = length;
	sregs.es    = FP_SEG(buffer);
	inregs.x.bx = FP_OFF(buffer);
	int86x(I2CINT,&inregs,&outregs,&sregs);

	if(outregs.x.flags & 0x01)
	{
		return outregs.h.al;
	}
	return 0;
}


int I2C_receive_block (unsigned char slave, char far * buffer, int length)
{
	inregs.h.ah = 0x83;
	inregs.h.al = slave | 0x01;
	inregs.x.cx = length;
	sregs.es    = FP_SEG(buffer);
	inregs.x.bx = FP_OFF(buffer);
	int86x(I2CINT,&inregs,&outregs,&sregs);

	if(outregs.x.flags & 0x01)
	{
		return outregs.h.al;
	}
	return 0;
}


int I2C_transmit_char (unsigned char slave, char c)
{
	inregs.h.ah = 0x82;
	inregs.h.al = slave & 0xFE;
	inregs.h.cl = c;
	int86(I2CINT,&inregs,&outregs);

	if(outregs.x.flags&0x01)
	{
		return (int)outregs.h.al & 0x00FF;
	}
	return 0;
}


int I2C_receive_char (unsigned char slave, char * c, unsigned char lastchar)
{
	inregs.h.ah = 0x82;
	inregs.h.al = slave | 0x01;
	if(lastchar)
	{
	  inregs.h.cl = 1;     //want more characters
	}
	else
	{
	  inregs.h.cl = 0;     //only one character
	}
	int86(I2CINT,&inregs,&outregs);

	if(outregs.x.flags&0x01)
	{
		*c = 0;
		return (int)outregs.h.al & 0x00FF;
	}
	*c = (char)outregs.h.ch;
	return 0;
}

/****************************************************************************/
//main
/****************************************************************************/

void main (void)
{
   unsigned char tmp=0x02;
   static char c = 0x0;
   unsigned result;

   printf("\nBG10 I2C bus test\n");

   //init i2c bus
   printf("\nInit I2C bus...");
   I2C_init(0);

   //scan for EEPROM_SLAVE
   printf("\nScanning I2C bus for BG10 EEPROM slave 0x%02X", EEPROM_SLAVE );
   result = I2C_scan(2, 0xFE);

   if(result != EEPROM_SLAVE)
   {
     printf("\nI2C scan error %02Xh\n",result);
     return;
   }
   printf("\nEEPROM slave 0x%02X found\n",outregs.h.al);


   printf("\n\nSending text 'Hello' to address C0h...");

   I2C_release();

   I2C_transmit_char(EEPROM_SLAVE,12);

   I2C_release();

   I2C_transmit_char(EEPROM_SLAVE,'H');
   I2C_transmit_char(EEPROM_SLAVE,'e');
   I2C_transmit_char(EEPROM_SLAVE,'l');
   I2C_transmit_char(EEPROM_SLAVE,'l');
   I2C_transmit_char(EEPROM_SLAVE,'o');

   printf("\n\nSending text 'World' to address C0h...");

   I2C_release();

   I2C_transmit_block(EEPROM_SLAVE," World",6);

   I2C_release();

   printf("\n\nCalling text Nr. 1 from address C0h...");

   I2C_transmit_char(EEPROM_SLAVE,27);
   I2C_transmit_char(EEPROM_SLAVE,'D');
   I2C_transmit_char(EEPROM_SLAVE,2);

   I2C_release();

   printf("\n\nReading function keys...\n");

   while(1)
   {
      api_sleep(100);

      tmp=I2C_receive_block(EEPROM_SLAVE,&c,1);
      if (tmp==0)
      {
         printf("Read: %02Xh ",(unsigned char)c);
      }
      else
      {
         printf("Error: %u ",tmp);
      }

      I2C_release();
   }
}
